<?php

namespace App\Http\Controllers;

use App\Models\Income;
use App\Models\Income_category;
use App\Models\Income_detail;
use App\Models\Transaction;
use App\Models\Transaction_detail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;


class IncomeController extends Controller
{

    public function show(Request $request)
    {
        $query = Income_detail::with('income_category');

        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        if ($request->filled('date')) {
            $query->whereDate('date', $request->date);
        }

        if ($request->filled('category')) {
            $query->where('income_account_id', $request->category);
        }

        $incomeDetails = $query->latest()->paginate(10)->withQueryString();

        return view('income.all-income', [
            'incomeDetails' => $incomeDetails,
            'categories' => Income_category::all()
        ]);
    }


    public function pdfIncome(Request $request)
    {
        $query = Income_detail::with('income_category');

        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        if ($request->filled('date')) {
            $query->whereDate('date', $request->date);
        }

        if ($request->filled('category')) {
            $query->where('income_account_id', $request->category);
        }

        $incomeDetails = $query->latest()->get();

        $pdf = PDF::loadView('income.pdfIncome', compact('incomeDetails'))->setPaper('a4', 'landscape');
        return $pdf->download('income_report.pdf');
    }




    public function store(Request $request)
    {
        $validated = $request->validate([
            'incomeCatg' => 'required|string|max:255',
            'date' => 'required|date',
            'amount' => 'required|numeric',
            'desc' => 'nullable|string',
            'fileInput' => 'nullable|file|mimes:jpg,jpeg,png,pdf,docx',
        ]);

        DB::beginTransaction();

        try {
            // Handle file upload if exists
            $filePath = null;
            if ($request->hasFile('fileInput')) {
                $filePath = $request->file('fileInput')->store('fileInput', 'public');
            }

            // Generate Voucher Number
            $id = Income::max('id');
            $voucherNumber = 'OIV-' . ($id == null ? 1 : $id + 1);

            // Create Income
            $income = Income::create([
                'date' => $request->date,
                'voucher_number' => $voucherNumber,
                'description' => $request->desc,
                'amount' => $request->amount,
                'status' => 1
            ]);

            // Create Income Detail
            Income_detail::create([
                'income_id' => $income->id,
                'income_account_id' => $request->incomeCatg,
                'description' => $request->desc,
                'amount' => $request->amount,
                'status' => 1,
                'date' => $request->date,
                'receipt_file' => $filePath ?? "N/A"
            ]);

            // Create Transaction
            $transaction = Transaction::create([
                'transaction_type' => 'Other Income',
                'trans_type_id' => $income->id,
                'transaction_date' => now(),
                'total_amount' => $request->amount,
                'remarks' => 'other Income'
            ]);

            // Other Income Category used here
            Transaction_detail::create([
                'transaction_id' => $transaction->id,
                'coa_id'       => $request->incomeCatg,
                'transaction_type'         => "Other Income",
                'debit'         => 0,
                'credit'         => $request->amount,
            ]);

            // Debit other income to cash in hand headCode 20500
            Transaction_detail::create([
                'transaction_id' => $transaction->id,
                'coa_id'       => '20500',
                'transaction_type'         => "Other Income",
                'debit'         => $request->amount,
                'credit'         => 0,
            ]);

            DB::commit();

            return redirect()->back()->with('success', 'Income created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            // Delete uploaded file if it exists and transaction failed
            if (isset($filePath) && Storage::disk('public')->exists($filePath)) {
                Storage::disk('public')->delete($filePath);
            }

            return redirect()->back()->with('error', 'Failed to create income. Error: ' . $e->getMessage());
        }
    }
}
